package models

import (
	"encoding/json"
	"testing"
)

func TestRecord_GetSet(t *testing.T) {
	record := NewRecord()

	// Test Set and Get
	record.Set("name", "Alice")
	record.Set("age", 30)
	record.Set("score", 95.5)

	name, ok := record.Get("name")
	if !ok || name != "Alice" {
		t.Errorf("Expected name=Alice, got %v", name)
	}

	age, ok := record.Get("age")
	if !ok || age != 30 {
		t.Errorf("Expected age=30, got %v", age)
	}
}

func TestRecord_GetString(t *testing.T) {
	record := NewRecord()
	record.Set("name", "Bob")
	record.Set("age", 25)

	tests := []struct {
		field    string
		expected string
		hasError bool
	}{
		{"name", "Bob", false},
		{"age", "25", false},
		{"missing", "", true},
	}

	for _, tt := range tests {
		result, err := record.GetString(tt.field)
		if tt.hasError {
			if err == nil {
				t.Errorf("Expected error for field %s", tt.field)
			}
		} else {
			if err != nil {
				t.Errorf("Unexpected error for field %s: %v", tt.field, err)
			}
			if result != tt.expected {
				t.Errorf("Expected %s, got %s", tt.expected, result)
			}
		}
	}
}

func TestRecord_GetInt(t *testing.T) {
	record := NewRecord()
	record.Set("count", 42)
	record.Set("float", 10.5)
	record.Set("str", "100")

	tests := []struct {
		field    string
		expected int
		hasError bool
	}{
		{"count", 42, false},
		{"float", 10, false},
		{"str", 100, false},
		{"missing", 0, true},
	}

	for _, tt := range tests {
		result, err := record.GetInt(tt.field)
		if tt.hasError {
			if err == nil {
				t.Errorf("Expected error for field %s", tt.field)
			}
		} else {
			if err != nil {
				t.Errorf("Unexpected error for field %s: %v", tt.field, err)
			}
			if result != tt.expected {
				t.Errorf("Expected %d, got %d", tt.expected, result)
			}
		}
	}
}

func TestRecord_GetFloat(t *testing.T) {
	record := NewRecord()
	record.Set("price", 29.99)
	record.Set("count", 10)
	record.Set("str", "15.5")

	tests := []struct {
		field    string
		expected float64
		hasError bool
	}{
		{"price", 29.99, false},
		{"count", 10.0, false},
		{"str", 15.5, false},
		{"missing", 0, true},
	}

	for _, tt := range tests {
		result, err := record.GetFloat(tt.field)
		if tt.hasError {
			if err == nil {
				t.Errorf("Expected error for field %s", tt.field)
			}
		} else {
			if err != nil {
				t.Errorf("Unexpected error for field %s: %v", tt.field, err)
			}
			if result != tt.expected {
				t.Errorf("Expected %f, got %f", tt.expected, result)
			}
		}
	}
}

func TestRecord_JSON(t *testing.T) {
	record := NewRecord()
	record.Set("name", "Charlie")
	record.Set("age", 35)
	record.Set("score", 88.5)

	// Marshal
	data, err := json.Marshal(record)
	if err != nil {
		t.Fatalf("Failed to marshal: %v", err)
	}

	// Unmarshal
	newRecord := NewRecord()
	if err := json.Unmarshal(data, newRecord); err != nil {
		t.Fatalf("Failed to unmarshal: %v", err)
	}

	// Verify fields
	if name, _ := newRecord.GetString("name"); name != "Charlie" {
		t.Errorf("Expected name=Charlie, got %s", name)
	}

	if age, _ := newRecord.GetFloat("age"); age != 35.0 {
		t.Errorf("Expected age=35, got %f", age)
	}
}
