package models

import (
	"fmt"
)

// FieldType represents a field data type
type FieldType string

const (
	TypeString FieldType = "string"
	TypeInt    FieldType = "int"
	TypeFloat  FieldType = "float"
	TypeBool   FieldType = "bool"
)

// Schema defines the structure of records
type Schema struct {
	Fields map[string]FieldType
}

// NewSchema creates a new schema
func NewSchema() *Schema {
	return &Schema{
		Fields: make(map[string]FieldType),
	}
}

// AddField adds a field to the schema
func (s *Schema) AddField(name string, fieldType FieldType) {
	s.Fields[name] = fieldType
}

// Validate validates a record against the schema
func (s *Schema) Validate(record *Record) error {
	for fieldName, fieldType := range s.Fields {
		val, ok := record.Get(fieldName)
		if !ok {
			return fmt.Errorf("missing required field: %s", fieldName)
		}

		if err := s.validateType(fieldName, val, fieldType); err != nil {
			return err
		}
	}

	return nil
}

func (s *Schema) validateType(fieldName string, value interface{}, expectedType FieldType) error {
	switch expectedType {
	case TypeString:
		if _, ok := value.(string); !ok {
			return fmt.Errorf("field %s: expected string, got %T", fieldName, value)
		}
	case TypeInt:
		switch value.(type) {
		case int, int64:
			return nil
		default:
			return fmt.Errorf("field %s: expected int, got %T", fieldName, value)
		}
	case TypeFloat:
		switch value.(type) {
		case float64, int, int64:
			return nil
		default:
			return fmt.Errorf("field %s: expected float, got %T", fieldName, value)
		}
	case TypeBool:
		if _, ok := value.(bool); !ok {
			return fmt.Errorf("field %s: expected bool, got %T", fieldName, value)
		}
	default:
		return fmt.Errorf("unknown field type: %s", expectedType)
	}

	return nil
}
