package main

import (
	"context"
	"log"
	"net/http"
	"os"
	"os/signal"
	"syscall"
	"time"

	"github.com/example/jobqueue/internal/api"
	"github.com/example/jobqueue/internal/queue"
	"github.com/gin-gonic/gin"
	"github.com/prometheus/client_golang/prometheus/promhttp"
)

func main() {
	// Get Redis address from environment
	redisAddr := os.Getenv("REDIS_ADDR")
	if redisAddr == "" {
		redisAddr = "localhost:6379"
	}

	// Create queue
	q, err := queue.NewRedisQueue(redisAddr)
	if err != nil {
		log.Fatalf("Failed to create queue: %v", err)
	}
	defer q.Close()

	// Create API handler
	handler := api.NewHandler(q)

	// Setup router
	router := gin.Default()

	// Serve static files
	router.Static("/static", "./web/dashboard")
	router.GET("/", func(c *gin.Context) {
		c.File("./web/dashboard/index.html")
	})

	// API routes
	v1 := router.Group("/api/v1")
	{
		v1.POST("/jobs", handler.SubmitJob)
		v1.GET("/jobs/:id", handler.GetJobStatus)
		v1.GET("/stats", handler.GetStats)
		v1.GET("/health", handler.HealthCheck)
	}

	// Metrics endpoint
	router.GET("/metrics", gin.WrapH(promhttp.Handler()))

	// Start server
	srv := &http.Server{
		Addr:    ":8080",
		Handler: router,
	}

	go func() {
		log.Println("API server listening on :8080")
		if err := srv.ListenAndServe(); err != nil && err != http.ErrServerClosed {
			log.Fatalf("Server error: %v", err)
		}
	}()

	// Wait for interrupt signal
	quit := make(chan os.Signal, 1)
	signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
	<-quit

	log.Println("Shutting down server...")

	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()

	if err := srv.Shutdown(ctx); err != nil {
		log.Fatalf("Server forced to shutdown: %v", err)
	}

	log.Println("Server exited")
}
