package main

import (
	"log"
	"os"
	"os/signal"
	"strconv"
	"syscall"

	"github.com/example/jobqueue/internal/jobs"
	"github.com/example/jobqueue/internal/metrics"
	"github.com/example/jobqueue/internal/queue"
	"github.com/example/jobqueue/internal/worker"
)

func main() {
	// Get configuration from environment
	redisAddr := os.Getenv("REDIS_ADDR")
	if redisAddr == "" {
		redisAddr = "localhost:6379"
	}

	workerCount := 4
	if count := os.Getenv("WORKER_COUNT"); count != "" {
		if n, err := strconv.Atoi(count); err == nil {
			workerCount = n
		}
	}

	// Create queue
	q, err := queue.NewRedisQueue(redisAddr)
	if err != nil {
		log.Fatalf("Failed to create queue: %v", err)
	}
	defer q.Close()

	// Create job factory
	factory := jobs.NewFactory()

	// Create metrics collector
	metricsCollector := metrics.NewCollector()

	// Create worker pool
	pool := worker.NewPool(workerCount, q, factory, metricsCollector)

	// Start workers
	pool.Start()

	// Wait for interrupt signal
	quit := make(chan os.Signal, 1)
	signal.Notify(quit, syscall.SIGINT, syscall.SIGTERM)
	<-quit

	// Graceful shutdown
	pool.Stop()

	log.Println("Workers stopped")
}
