package api

import (
	"encoding/json"
	"net/http"
	"time"

	"github.com/example/jobqueue/internal/jobs"
	"github.com/example/jobqueue/internal/queue"
	"github.com/gin-gonic/gin"
	"github.com/google/uuid"
)

// Handler handles HTTP requests
type Handler struct {
	queue queue.Queue
}

// NewHandler creates a new API handler
func NewHandler(q queue.Queue) *Handler {
	return &Handler{queue: q}
}

// SubmitJobRequest represents a job submission request
type SubmitJobRequest struct {
	Type        string          `json:"type" binding:"required"`
	Payload     json.RawMessage `json:"payload" binding:"required"`
	MaxAttempts int             `json:"max_attempts,omitempty"`
	Priority    int             `json:"priority,omitempty"`
}

// SubmitJobResponse represents a job submission response
type SubmitJobResponse struct {
	JobID string `json:"job_id"`
}

// SubmitJob handles job submission
func (h *Handler) SubmitJob(c *gin.Context) {
	var req SubmitJobRequest
	if err := c.ShouldBindJSON(&req); err != nil {
		c.JSON(http.StatusBadRequest, gin.H{"error": err.Error()})
		return
	}

	// Validate job type
	validTypes := map[string]bool{"email": true, "data_process": true}
	if !validTypes[req.Type] {
		c.JSON(http.StatusBadRequest, gin.H{
			"error": "invalid job type, must be one of: email, data_process",
		})
		return
	}

	// Set defaults
	if req.MaxAttempts == 0 {
		req.MaxAttempts = 3
	}
	if req.Priority == 0 {
		req.Priority = 5
	}

	// Create envelope
	envelope := &jobs.Envelope{
		ID:          uuid.New().String(),
		Type:        req.Type,
		Payload:     req.Payload,
		CreatedAt:   time.Now(),
		Attempts:    0,
		MaxAttempts: req.MaxAttempts,
		Priority:    req.Priority,
	}

	// Enqueue
	if err := h.queue.Enqueue(c.Request.Context(), envelope); err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": err.Error()})
		return
	}

	c.JSON(http.StatusCreated, SubmitJobResponse{JobID: envelope.ID})
}

// GetJobStatus retrieves job status
func (h *Handler) GetJobStatus(c *gin.Context) {
	jobID := c.Param("id")

	result, err := h.queue.GetResult(c.Request.Context(), jobID)
	if err != nil {
		c.JSON(http.StatusNotFound, gin.H{"error": "job not found or not completed"})
		return
	}

	c.JSON(http.StatusOK, result)
}

// GetStats retrieves queue statistics
func (h *Handler) GetStats(c *gin.Context) {
	stats, err := h.queue.Stats(c.Request.Context())
	if err != nil {
		c.JSON(http.StatusInternalServerError, gin.H{"error": err.Error()})
		return
	}

	c.JSON(http.StatusOK, stats)
}

// HealthCheck handles health check requests
func (h *Handler) HealthCheck(c *gin.Context) {
	c.JSON(http.StatusOK, gin.H{
		"status": "healthy",
		"time":   time.Now(),
	})
}
