package jobs

import (
	"context"
	"fmt"
	"time"
)

// DataProcessPayload contains data for processing operations
type DataProcessPayload struct {
	SourceURL   string            `json:"source_url"`
	Operation   string            `json:"operation"` // transform, aggregate, export
	Filters     map[string]string `json:"filters,omitempty"`
	Destination string            `json:"destination,omitempty"`
}

// DataProcessJob handles data processing tasks
type DataProcessJob struct {
	id      string
	payload DataProcessPayload
}

// NewDataProcessJob creates a new data processing job
func NewDataProcessJob(id string) *DataProcessJob {
	return &DataProcessJob{
		id: id,
	}
}

func (j *DataProcessJob) ID() string {
	return j.id
}

func (j *DataProcessJob) Type() string {
	return "data_process"
}

func (j *DataProcessJob) GetPayload() DataProcessPayload {
	return j.payload
}

func (j *DataProcessJob) SetPayload(payload DataProcessPayload) {
	j.payload = payload
}

func (j *DataProcessJob) Execute(ctx context.Context, payload DataProcessPayload) error {
	// Simulate data processing
	steps := []string{
		"Fetching data from " + payload.SourceURL,
		"Applying operation: " + payload.Operation,
		"Processing with filters",
	}

	if payload.Destination != "" {
		steps = append(steps, "Exporting to "+payload.Destination)
	}

	for i, step := range steps {
		select {
		case <-time.After(1 * time.Second):
			fmt.Printf("📊 [%d/%d] %s\n", i+1, len(steps), step)
		case <-ctx.Done():
			return ctx.Err()
		}
	}

	return nil
}

// Validate checks if the data process payload is valid
func (p DataProcessPayload) Validate() error {
	if p.SourceURL == "" {
		return fmt.Errorf("source URL is required")
	}
	validOps := map[string]bool{
		"transform": true,
		"aggregate": true,
		"export":    true,
	}
	if !validOps[p.Operation] {
		return fmt.Errorf("invalid operation: %s", p.Operation)
	}
	return nil
}
