package jobs

import (
	"context"
	"fmt"
	"time"
)

// EmailPayload contains data for sending an email
type EmailPayload struct {
	To      string `json:"to"`
	Subject string `json:"subject"`
	Body    string `json:"body"`
	From    string `json:"from"`
}

// EmailJob handles email sending
type EmailJob struct {
	id      string
	payload EmailPayload
}

// NewEmailJob creates a new email job
func NewEmailJob(id string) *EmailJob {
	return &EmailJob{
		id: id,
	}
}

func (j *EmailJob) ID() string {
	return j.id
}

func (j *EmailJob) Type() string {
	return "email"
}

func (j *EmailJob) GetPayload() EmailPayload {
	return j.payload
}

func (j *EmailJob) SetPayload(payload EmailPayload) {
	j.payload = payload
}

func (j *EmailJob) Execute(ctx context.Context, payload EmailPayload) error {
	// Simulate email sending
	select {
	case <-time.After(2 * time.Second):
		// In a real implementation, this would use an SMTP library
		fmt.Printf("📧 Sending email to %s: %s\n", payload.To, payload.Subject)
		return nil
	case <-ctx.Done():
		return ctx.Err()
	}
}

// Validate checks if the email payload is valid
func (p EmailPayload) Validate() error {
	if p.To == "" {
		return fmt.Errorf("recipient email is required")
	}
	if p.Subject == "" {
		return fmt.Errorf("subject is required")
	}
	if p.Body == "" {
		return fmt.Errorf("body is required")
	}
	return nil
}
