package jobs

import (
	"encoding/json"
	"fmt"
	"sync"
)

// JobFactory implements the Factory pattern for creating jobs
type JobFactory struct {
	mu       sync.RWMutex
	creators map[string]func(id string) interface{}
}

// NewFactory creates a new job factory
func NewFactory() *JobFactory {
	f := &JobFactory{
		creators: make(map[string]func(id string) interface{}),
	}

	// Register built-in job types
	f.Register("email", func(id string) interface{} {
		return NewEmailJob(id)
	})
	f.Register("data_process", func(id string) interface{} {
		return NewDataProcessJob(id)
	})

	return f
}

// Register adds a new job type creator
func (f *JobFactory) Register(jobType string, creator func(id string) interface{}) {
	f.mu.Lock()
	defer f.mu.Unlock()
	f.creators[jobType] = creator
}

// Create instantiates a job from an envelope
func (f *JobFactory) Create(envelope *Envelope) (interface{}, error) {
	f.mu.RLock()
	creator, exists := f.creators[envelope.Type]
	f.mu.RUnlock()

	if !exists {
		return nil, fmt.Errorf("unknown job type: %s", envelope.Type)
	}

	job := creator(envelope.ID)

	// Unmarshal payload into the job
	switch envelope.Type {
	case "email":
		emailJob := job.(*EmailJob)
		var payload EmailPayload
		if err := json.Unmarshal(envelope.Payload, &payload); err != nil {
			return nil, fmt.Errorf("failed to unmarshal email payload: %w", err)
		}
		emailJob.SetPayload(payload)
		return emailJob, nil

	case "data_process":
		dataJob := job.(*DataProcessJob)
		var payload DataProcessPayload
		if err := json.Unmarshal(envelope.Payload, &payload); err != nil {
			return nil, fmt.Errorf("failed to unmarshal data process payload: %w", err)
		}
		dataJob.SetPayload(payload)
		return dataJob, nil

	default:
		return nil, fmt.Errorf("unhandled job type in create: %s", envelope.Type)
	}
}
