package jobs

import (
	"context"
	"encoding/json"
	"time"
)

// Status represents the current state of a job
type Status string

const (
	StatusPending   Status = "pending"
	StatusRunning   Status = "running"
	StatusCompleted Status = "completed"
	StatusFailed    Status = "failed"
	StatusRetrying  Status = "retrying"
)

// Job is a generic interface for all job types
type Job[T any] interface {
	// ID returns the unique identifier for this job
	ID() string
	// Type returns the job type name
	Type() string
	// Execute processes the job with the given payload
	Execute(ctx context.Context, payload T) error
	// GetPayload returns the job payload
	GetPayload() T
	// SetPayload updates the job payload
	SetPayload(payload T)
}

// Result stores the outcome of a job execution
type Result struct {
	JobID       string                 `json:"job_id"`
	Status      Status                 `json:"status"`
	Result      interface{}            `json:"result,omitempty"`
	Error       string                 `json:"error,omitempty"`
	StartedAt   time.Time              `json:"started_at"`
	CompletedAt time.Time              `json:"completed_at,omitempty"`
	Attempts    int                    `json:"attempts"`
	Metadata    map[string]interface{} `json:"metadata,omitempty"`
}

// Envelope wraps a job with metadata for queue processing
type Envelope struct {
	ID          string          `json:"id"`
	Type        string          `json:"type"`
	Payload     json.RawMessage `json:"payload"`
	CreatedAt   time.Time       `json:"created_at"`
	ScheduledAt time.Time       `json:"scheduled_at,omitempty"`
	Attempts    int             `json:"attempts"`
	MaxAttempts int             `json:"max_attempts"`
	Priority    int             `json:"priority"`
}

// Factory creates job instances from envelopes
type Factory interface {
	Create(envelope *Envelope) (interface{}, error)
	Register(jobType string, creator func(id string) interface{})
}
