package metrics

import (
	"time"

	"github.com/prometheus/client_golang/prometheus"
	"github.com/prometheus/client_golang/prometheus/promauto"
)

// Collector collects job metrics
type Collector struct {
	jobsStarted   *prometheus.CounterVec
	jobsCompleted *prometheus.CounterVec
	jobsFailed    *prometheus.CounterVec
	jobDuration   *prometheus.HistogramVec
}

// NewCollector creates a new metrics collector
func NewCollector() *Collector {
	return &Collector{
		jobsStarted: promauto.NewCounterVec(
			prometheus.CounterOpts{
				Name: "jobqueue_jobs_started_total",
				Help: "Total number of jobs started",
			},
			[]string{"job_type"},
		),
		jobsCompleted: promauto.NewCounterVec(
			prometheus.CounterOpts{
				Name: "jobqueue_jobs_completed_total",
				Help: "Total number of jobs completed successfully",
			},
			[]string{"job_type"},
		),
		jobsFailed: promauto.NewCounterVec(
			prometheus.CounterOpts{
				Name: "jobqueue_jobs_failed_total",
				Help: "Total number of jobs failed",
			},
			[]string{"job_type"},
		),
		jobDuration: promauto.NewHistogramVec(
			prometheus.HistogramOpts{
				Name:    "jobqueue_job_duration_seconds",
				Help:    "Job execution duration in seconds",
				Buckets: prometheus.DefBuckets,
			},
			[]string{"job_type", "status"},
		),
	}
}

// JobStarted records a job start
func (c *Collector) JobStarted(jobType string) {
	c.jobsStarted.WithLabelValues(jobType).Inc()
}

// JobCompleted records a successful job completion
func (c *Collector) JobCompleted(jobType string, duration time.Duration) {
	c.jobsCompleted.WithLabelValues(jobType).Inc()
	c.jobDuration.WithLabelValues(jobType, "completed").Observe(duration.Seconds())
}

// JobFailed records a job failure
func (c *Collector) JobFailed(jobType string, duration time.Duration) {
	c.jobsFailed.WithLabelValues(jobType).Inc()
	c.jobDuration.WithLabelValues(jobType, "failed").Observe(duration.Seconds())
}
