package queue

import (
	"context"
	"time"

	"github.com/example/jobqueue/internal/jobs"
)

// Queue defines the interface for job queue operations
type Queue interface {
	// Enqueue adds a job to the queue
	Enqueue(ctx context.Context, envelope *jobs.Envelope) error

	// Dequeue retrieves the next job from the queue (blocking)
	Dequeue(ctx context.Context, timeout time.Duration) (*jobs.Envelope, error)

	// Complete marks a job as completed
	Complete(ctx context.Context, jobID string, result *jobs.Result) error

	// Fail marks a job as failed and potentially requeues it
	Fail(ctx context.Context, jobID string, err error, retry bool) error

	// GetResult retrieves the result of a completed job
	GetResult(ctx context.Context, jobID string) (*jobs.Result, error)

	// Stats returns queue statistics
	Stats(ctx context.Context) (*Stats, error)

	// Close closes the queue connection
	Close() error
}

// Stats contains queue statistics
type Stats struct {
	Pending   int64 `json:"pending"`
	Running   int64 `json:"running"`
	Completed int64 `json:"completed"`
	Failed    int64 `json:"failed"`
	Total     int64 `json:"total"`
}
