package queue

import (
	"context"
	"encoding/json"
	"fmt"
	"time"

	"github.com/example/jobqueue/internal/jobs"
	"github.com/redis/go-redis/v9"
)

const (
	queueKey        = "jobqueue:pending"
	runningKey      = "jobqueue:running"
	completedKey    = "jobqueue:completed"
	failedKey       = "jobqueue:failed"
	resultsKeyFmt   = "jobqueue:result:%s"
	jobKeyFmt       = "jobqueue:job:%s"
	statsKey        = "jobqueue:stats"
)

// RedisQueue implements Queue using Redis
type RedisQueue struct {
	client *redis.Client
}

// NewRedisQueue creates a new Redis-backed queue
func NewRedisQueue(addr string) (*RedisQueue, error) {
	client := redis.NewClient(&redis.Options{
		Addr:         addr,
		DialTimeout:  5 * time.Second,
		ReadTimeout:  3 * time.Second,
		WriteTimeout: 3 * time.Second,
		PoolSize:     10,
	})

	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()

	if err := client.Ping(ctx).Err(); err != nil {
		return nil, fmt.Errorf("failed to connect to Redis: %w", err)
	}

	return &RedisQueue{client: client}, nil
}

func (q *RedisQueue) Enqueue(ctx context.Context, envelope *jobs.Envelope) error {
	data, err := json.Marshal(envelope)
	if err != nil {
		return fmt.Errorf("failed to marshal envelope: %w", err)
	}

	pipe := q.client.Pipeline()

	// Add to pending queue with priority
	score := float64(envelope.Priority)
	if !envelope.ScheduledAt.IsZero() {
		score = float64(envelope.ScheduledAt.Unix())
	}
	pipe.ZAdd(ctx, queueKey, redis.Z{Score: score, Member: envelope.ID})

	// Store job data
	pipe.Set(ctx, fmt.Sprintf(jobKeyFmt, envelope.ID), data, 24*time.Hour)

	// Update stats
	pipe.HIncrBy(ctx, statsKey, "total", 1)
	pipe.HIncrBy(ctx, statsKey, "pending", 1)

	_, err = pipe.Exec(ctx)
	return err
}

func (q *RedisQueue) Dequeue(ctx context.Context, timeout time.Duration) (*jobs.Envelope, error) {
	// Use BZPOPMIN for blocking dequeue with timeout
	deadline := time.Now().Add(timeout)
	remaining := time.Until(deadline)

	for remaining > 0 {
		result, err := q.client.BZPopMin(ctx, remaining, queueKey).Result()
		if err != nil {
			if err == redis.Nil {
				return nil, nil // Timeout
			}
			return nil, err
		}

		jobID := result.Member.(string)

		// Get job data
		data, err := q.client.Get(ctx, fmt.Sprintf(jobKeyFmt, jobID)).Bytes()
		if err != nil {
			if err == redis.Nil {
				// Job data expired, continue to next
				remaining = time.Until(deadline)
				continue
			}
			return nil, err
		}

		var envelope jobs.Envelope
		if err := json.Unmarshal(data, &envelope); err != nil {
			return nil, fmt.Errorf("failed to unmarshal envelope: %w", err)
		}

		// Move to running
		pipe := q.client.Pipeline()
		pipe.SAdd(ctx, runningKey, jobID)
		pipe.HIncrBy(ctx, statsKey, "pending", -1)
		pipe.HIncrBy(ctx, statsKey, "running", 1)
		if _, err := pipe.Exec(ctx); err != nil {
			return nil, err
		}

		return &envelope, nil
	}

	return nil, nil
}

func (q *RedisQueue) Complete(ctx context.Context, jobID string, result *jobs.Result) error {
	data, err := json.Marshal(result)
	if err != nil {
		return fmt.Errorf("failed to marshal result: %w", err)
	}

	pipe := q.client.Pipeline()

	// Store result
	pipe.Set(ctx, fmt.Sprintf(resultsKeyFmt, jobID), data, 24*time.Hour)

	// Move from running to completed
	pipe.SRem(ctx, runningKey, jobID)
	pipe.SAdd(ctx, completedKey, jobID)

	// Update stats
	pipe.HIncrBy(ctx, statsKey, "running", -1)
	pipe.HIncrBy(ctx, statsKey, "completed", 1)

	_, err = pipe.Exec(ctx)
	return err
}

func (q *RedisQueue) Fail(ctx context.Context, jobID string, jobErr error, retry bool) error {
	pipe := q.client.Pipeline()

	if retry {
		// Get job and increment attempts
		data, err := q.client.Get(ctx, fmt.Sprintf(jobKeyFmt, jobID)).Bytes()
		if err == nil {
			var envelope jobs.Envelope
			if json.Unmarshal(data, &envelope) == nil {
				envelope.Attempts++
				if envelope.Attempts < envelope.MaxAttempts {
					// Requeue with exponential backoff
					backoff := time.Duration(envelope.Attempts*envelope.Attempts) * time.Second
					envelope.ScheduledAt = time.Now().Add(backoff)

					newData, _ := json.Marshal(envelope)
					pipe.Set(ctx, fmt.Sprintf(jobKeyFmt, jobID), newData, 24*time.Hour)
					pipe.ZAdd(ctx, queueKey, redis.Z{
						Score:  float64(envelope.ScheduledAt.Unix()),
						Member: jobID,
					})
					pipe.SRem(ctx, runningKey, jobID)
					pipe.HIncrBy(ctx, statsKey, "running", -1)
					pipe.HIncrBy(ctx, statsKey, "pending", 1)

					_, err := pipe.Exec(ctx)
					return err
				}
			}
		}
	}

	// Mark as failed
	result := &jobs.Result{
		JobID:       jobID,
		Status:      jobs.StatusFailed,
		Error:       jobErr.Error(),
		CompletedAt: time.Now(),
	}

	data, _ := json.Marshal(result)
	pipe.Set(ctx, fmt.Sprintf(resultsKeyFmt, jobID), data, 24*time.Hour)
	pipe.SRem(ctx, runningKey, jobID)
	pipe.SAdd(ctx, failedKey, jobID)
	pipe.HIncrBy(ctx, statsKey, "running", -1)
	pipe.HIncrBy(ctx, statsKey, "failed", 1)

	_, err := pipe.Exec(ctx)
	return err
}

func (q *RedisQueue) GetResult(ctx context.Context, jobID string) (*jobs.Result, error) {
	data, err := q.client.Get(ctx, fmt.Sprintf(resultsKeyFmt, jobID)).Bytes()
	if err != nil {
		if err == redis.Nil {
			return nil, fmt.Errorf("result not found for job %s", jobID)
		}
		return nil, err
	}

	var result jobs.Result
	if err := json.Unmarshal(data, &result); err != nil {
		return nil, fmt.Errorf("failed to unmarshal result: %w", err)
	}

	return &result, nil
}

func (q *RedisQueue) Stats(ctx context.Context) (*Stats, error) {
	data, err := q.client.HGetAll(ctx, statsKey).Result()
	if err != nil {
		return nil, err
	}

	stats := &Stats{}

	// Parse integers
	fmt.Sscanf(data["pending"], "%d", &stats.Pending)
	fmt.Sscanf(data["running"], "%d", &stats.Running)
	fmt.Sscanf(data["completed"], "%d", &stats.Completed)
	fmt.Sscanf(data["failed"], "%d", &stats.Failed)
	fmt.Sscanf(data["total"], "%d", &stats.Total)

	return stats, nil
}

func (q *RedisQueue) Close() error {
	return q.client.Close()
}
