package worker

import (
	"context"
	"fmt"
	"log"
	"sync"
	"time"

	"github.com/example/jobqueue/internal/jobs"
	"github.com/example/jobqueue/internal/metrics"
	"github.com/example/jobqueue/internal/queue"
)

// Pool manages a pool of workers
type Pool struct {
	size      int
	queue     queue.Queue
	factory   jobs.Factory
	metrics   *metrics.Collector
	wg        sync.WaitGroup
	ctx       context.Context
	cancel    context.CancelFunc
	observers []Observer
	mu        sync.RWMutex
}

// Observer watches job status changes
type Observer interface {
	OnJobStarted(jobID, jobType string)
	OnJobCompleted(jobID string, duration time.Duration)
	OnJobFailed(jobID string, err error)
}

// NewPool creates a new worker pool
func NewPool(size int, q queue.Queue, factory jobs.Factory, m *metrics.Collector) *Pool {
	ctx, cancel := context.WithCancel(context.Background())
	return &Pool{
		size:      size,
		queue:     q,
		factory:   factory,
		metrics:   m,
		ctx:       ctx,
		cancel:    cancel,
		observers: make([]Observer, 0),
	}
}

// AddObserver registers an observer
func (p *Pool) AddObserver(observer Observer) {
	p.mu.Lock()
	defer p.mu.Unlock()
	p.observers = append(p.observers, observer)
}

// Start starts all workers
func (p *Pool) Start() {
	for i := 0; i < p.size; i++ {
		p.wg.Add(1)
		go p.worker(i)
	}
	log.Printf("Started %d workers\n", p.size)
}

// Stop gracefully stops all workers
func (p *Pool) Stop() {
	log.Println("Stopping worker pool...")
	p.cancel()
	p.wg.Wait()
	log.Println("Worker pool stopped")
}

func (p *Pool) worker(id int) {
	defer p.wg.Done()

	log.Printf("Worker %d started\n", id)

	for {
		select {
		case <-p.ctx.Done():
			log.Printf("Worker %d stopping\n", id)
			return
		default:
			envelope, err := p.queue.Dequeue(p.ctx, 5*time.Second)
			if err != nil {
				log.Printf("Worker %d: dequeue error: %v\n", id, err)
				continue
			}

			if envelope == nil {
				// Timeout, continue
				continue
			}

			p.processJob(id, envelope)
		}
	}
}

func (p *Pool) processJob(workerID int, envelope *jobs.Envelope) {
	start := time.Now()

	// Notify observers
	p.notifyStarted(envelope.ID, envelope.Type)

	// Update metrics
	p.metrics.JobStarted(envelope.Type)

	log.Printf("Worker %d: processing job %s (type: %s, attempt: %d)\n",
		workerID, envelope.ID, envelope.Type, envelope.Attempts+1)

	// Create job instance
	job, err := p.factory.Create(envelope)
	if err != nil {
		p.handleError(envelope, fmt.Errorf("failed to create job: %w", err))
		return
	}

	// Execute job with timeout
	ctx, cancel := context.WithTimeout(p.ctx, 5*time.Minute)
	defer cancel()

	var execErr error

	// Execute based on job type
	switch envelope.Type {
	case "email":
		emailJob := job.(*jobs.EmailJob)
		execErr = emailJob.Execute(ctx, emailJob.GetPayload())
	case "data_process":
		dataJob := job.(*jobs.DataProcessJob)
		execErr = dataJob.Execute(ctx, dataJob.GetPayload())
	default:
		execErr = fmt.Errorf("unknown job type: %s", envelope.Type)
	}

	duration := time.Since(start)

	if execErr != nil {
		p.handleError(envelope, execErr)
		p.metrics.JobFailed(envelope.Type, duration)
		p.notifyFailed(envelope.ID, execErr)
		return
	}

	// Mark as completed
	result := &jobs.Result{
		JobID:       envelope.ID,
		Status:      jobs.StatusCompleted,
		StartedAt:   start,
		CompletedAt: time.Now(),
		Attempts:    envelope.Attempts + 1,
	}

	if err := p.queue.Complete(ctx, envelope.ID, result); err != nil {
		log.Printf("Worker %d: failed to mark job %s as completed: %v\n",
			workerID, envelope.ID, err)
	}

	p.metrics.JobCompleted(envelope.Type, duration)
	p.notifyCompleted(envelope.ID, duration)

	log.Printf("Worker %d: completed job %s in %v\n",
		workerID, envelope.ID, duration)
}

func (p *Pool) handleError(envelope *jobs.Envelope, err error) {
	log.Printf("Job %s failed: %v (attempt %d/%d)\n",
		envelope.ID, err, envelope.Attempts+1, envelope.MaxAttempts)

	retry := envelope.Attempts+1 < envelope.MaxAttempts

	ctx, cancel := context.WithTimeout(context.Background(), 5*time.Second)
	defer cancel()

	if err := p.queue.Fail(ctx, envelope.ID, err, retry); err != nil {
		log.Printf("Failed to handle job failure: %v\n", err)
	}
}

func (p *Pool) notifyStarted(jobID, jobType string) {
	p.mu.RLock()
	defer p.mu.RUnlock()
	for _, observer := range p.observers {
		observer.OnJobStarted(jobID, jobType)
	}
}

func (p *Pool) notifyCompleted(jobID string, duration time.Duration) {
	p.mu.RLock()
	defer p.mu.RUnlock()
	for _, observer := range p.observers {
		observer.OnJobCompleted(jobID, duration)
	}
}

func (p *Pool) notifyFailed(jobID string, err error) {
	p.mu.RLock()
	defer p.mu.RUnlock()
	for _, observer := range p.observers {
		observer.OnJobFailed(jobID, err)
	}
}
