package tests

import (
	"context"
	"encoding/json"
	"testing"
	"time"

	"github.com/example/jobqueue/internal/jobs"
	"github.com/example/jobqueue/internal/queue"
	"github.com/stretchr/testify/assert"
	"github.com/stretchr/testify/require"
)

func TestRedisQueueIntegration(t *testing.T) {
	if testing.Short() {
		t.Skip("Skipping integration test")
	}

	q, err := queue.NewRedisQueue("localhost:6379")
	require.NoError(t, err)
	defer q.Close()

	ctx := context.Background()

	// Create test envelope
	payload := jobs.EmailPayload{
		To:      "test@example.com",
		Subject: "Test",
		Body:    "Test body",
		From:    "noreply@example.com",
	}
	payloadData, _ := json.Marshal(payload)

	envelope := &jobs.Envelope{
		ID:          "test-job-1",
		Type:        "email",
		Payload:     payloadData,
		CreatedAt:   time.Now(),
		MaxAttempts: 3,
		Priority:    5,
	}

	// Enqueue
	err = q.Enqueue(ctx, envelope)
	assert.NoError(t, err)

	// Dequeue
	dequeued, err := q.Dequeue(ctx, 5*time.Second)
	assert.NoError(t, err)
	assert.NotNil(t, dequeued)
	assert.Equal(t, envelope.ID, dequeued.ID)
	assert.Equal(t, envelope.Type, dequeued.Type)

	// Complete
	result := &jobs.Result{
		JobID:       envelope.ID,
		Status:      jobs.StatusCompleted,
		CompletedAt: time.Now(),
	}
	err = q.Complete(ctx, envelope.ID, result)
	assert.NoError(t, err)

	// Get result
	retrieved, err := q.GetResult(ctx, envelope.ID)
	assert.NoError(t, err)
	assert.Equal(t, jobs.StatusCompleted, retrieved.Status)
}

func TestJobFactory(t *testing.T) {
	factory := jobs.NewFactory()

	// Create email job
	payload := jobs.EmailPayload{
		To:      "test@example.com",
		Subject: "Test",
		Body:    "Test body",
		From:    "noreply@example.com",
	}
	payloadData, _ := json.Marshal(payload)

	envelope := &jobs.Envelope{
		ID:      "test-1",
		Type:    "email",
		Payload: payloadData,
	}

	job, err := factory.Create(envelope)
	require.NoError(t, err)

	emailJob, ok := job.(*jobs.EmailJob)
	require.True(t, ok)
	assert.Equal(t, "test-1", emailJob.ID())
	assert.Equal(t, "email", emailJob.Type())
}

func BenchmarkEnqueue(b *testing.B) {
	q, err := queue.NewRedisQueue("localhost:6379")
	if err != nil {
		b.Skip("Redis not available")
	}
	defer q.Close()

	ctx := context.Background()
	payload, _ := json.Marshal(jobs.EmailPayload{
		To:      "test@example.com",
		Subject: "Benchmark",
		Body:    "Test",
		From:    "noreply@example.com",
	})

	b.ResetTimer()
	for i := 0; i < b.N; i++ {
		envelope := &jobs.Envelope{
			ID:          "bench-job",
			Type:        "email",
			Payload:     payload,
			CreatedAt:   time.Now(),
			MaxAttempts: 3,
			Priority:    5,
		}
		_ = q.Enqueue(ctx, envelope)
	}
}
