# Architecture Documentation

## Overview

This e-commerce platform demonstrates production microservices patterns using a simplified, educational approach. It's designed to be immediately runnable while showcasing real-world architectural decisions.

## System Architecture

### High-Level Architecture

```
                              ┌─────────────────┐
                              │                 │
                         ┌────┤  Client/Browser │
                         │    │                 │
                         │    └─────────────────┘
                         │
                         │    HTTP/JSON Requests
                         │
                         ▼
                ┌──────────────────┐
                │                  │
                │   API Gateway    │◄──── Entry point for all requests
                │   Port 8080      │      Routes to appropriate services
                │                  │      CORS, logging, recovery middleware
                └────────┬─────────┘
                         │
        ┌────────────────┼────────────────┬────────────────┐
        │                │                │                │
        ▼                ▼                ▼                ▼
┌──────────────┐ ┌──────────────┐ ┌──────────────┐ ┌──────────────┐
│   Product    │ │     User     │ │     Cart     │ │    Order     │
│   Service    │ │   Service    │ │   Service    │ │   Service    │
│   Port 8081  │ │   Port 8082  │ │   Port 8083  │ │   Port 8084  │
└──────────────┘ └──────────────┘ └──────────────┘ └──────────────┘
       │                │                │                │
       │                │                │                │
       └────────────────┴────────────────┴────────────────┘
                            │
                            ▼
                  ┌──────────────────┐
                  │   In-Memory DB   │
                  │  (Per Service)   │
                  └──────────────────┘
```

## Service Responsibilities

### 1. API Gateway (Port 8080)

**Purpose**: Single entry point for all client requests

**Responsibilities**:
- Request routing to backend services
- Load balancing (future)
- Authentication validation
- CORS handling
- Centralized logging
- Error recovery

**Endpoints**:
```
GET  /health              - Gateway health check
GET  /api/products        - List products (→ Product Service)
GET  /api/products/:id    - Get product (→ Product Service)
POST /api/auth/register   - Register user (→ User Service)
POST /api/auth/login      - Login (→ User Service)
GET  /api/cart            - Get cart (→ Cart Service)
POST /api/cart/items      - Add to cart (→ Cart Service)
POST /api/orders          - Create order (→ Order Service)
GET  /api/orders          - List orders (→ Order Service)
```

**Design Pattern**: API Gateway Pattern

**Key Code**:
```go
// Proxy pattern for routing
func proxyRequest(w http.ResponseWriter, r *http.Request, targetURL string, log *logger.Logger) {
    // Forward request to backend service
    // Handle response
}
```

### 2. Product Service (Port 8081)

**Purpose**: Product catalog management

**Responsibilities**:
- CRUD operations for products
- Product inventory tracking
- Product search (future)
- Price management

**Data Model**:
```go
type Product struct {
    ID          string
    Name        string
    Description string
    Price       float64
    SKU         string
    Inventory   int
    Category    string
    ImageURL    string
}
```

**Seeded Data**: 5 sample products on startup

**Design Pattern**: Repository Pattern (simplified)

### 3. User Service (Port 8082)

**Purpose**: User authentication and management

**Responsibilities**:
- User registration
- User authentication
- JWT token generation (simplified)
- Password hashing

**Data Model**:
```go
type User struct {
    ID        string
    Email     string
    Password  string  // SHA-256 hashed
    Name      string
    CreatedAt time.Time
}
```

**Security Notes**:
- Passwords hashed with SHA-256 (use bcrypt in production)
- JWT tokens simplified (use proper JWT library in production)
- Tokens expire in 24 hours

**Design Pattern**: Service Layer Pattern

### 4. Cart Service (Port 8083)

**Purpose**: Shopping cart management

**Responsibilities**:
- Add/remove items from cart
- Cart total calculation
- Guest cart support
- Inventory validation (via Product Service)

**Data Model**:
```go
type Cart struct {
    UserID string
    Items  []CartItem
    Total  float64
}

type CartItem struct {
    ProductID string
    Name      string
    Price     float64
    Quantity  int
}
```

**Inter-Service Communication**:
- Calls Product Service to validate products
- Checks inventory availability

**Design Pattern**: Aggregator Pattern

### 5. Order Service (Port 8084)

**Purpose**: Order processing and management

**Responsibilities**:
- Order creation
- Order history
- Order status tracking
- Payment processing (placeholder)

**Data Model**:
```go
type Order struct {
    ID              string
    UserID          string
    Items           []OrderItem
    Total           float64
    Status          string  // pending, confirmed, shipped, delivered
    ShippingAddress Address
    PaymentMethod   string
    CreatedAt       time.Time
    UpdatedAt       time.Time
}
```

**Order Flow**:
1. Receive order request
2. Validate items
3. Calculate total
4. Create order (status: pending)
5. Process payment (simplified)
6. Update status (confirmed)
7. Return order details

**Design Pattern**: State Pattern (order status)

## Cross-Cutting Concerns

### Middleware Stack

All services use a middleware stack:

```go
handler := middleware.CORS(
    middleware.Recovery(log)(
        middleware.Logger(log)(mux),
    ),
)
```

**Middleware Layers**:
1. **CORS**: Cross-origin resource sharing
2. **Recovery**: Panic recovery
3. **Logger**: Request/response logging

Gateway adds:
4. **Auth**: JWT validation (for protected routes)

### Configuration Management

**Pattern**: Environment-based configuration with defaults

```go
type Config struct {
    ServiceName string
    Port        string
    JWTSecret   string

    // Service URLs
    ProductURL  string
    UserURL     string
    CartURL     string
    OrderURL    string
}
```

**Configuration Sources**:
1. Environment variables (production)
2. Defaults (development)

### Logging

**Pattern**: Structured logging with prefixes

```go
log := logger.New("[SERVICE_NAME]")
log.Info("Message: %s", value)
log.Error("Error: %v", err)
```

**Log Format**:
```
[SERVICE_NAME] 2025/10/21 13:00:00 file.go:42: [LEVEL] Message
```

### Data Persistence

**Current**: In-memory database (per service)

**Production Alternative**:
```
┌──────────────┐
│   Product    │──────┐
│   Service    │      │
└──────────────┘      │
                      ├──► PostgreSQL (Products DB)
┌──────────────┐      │
│    Cart      │──────┘
│   Service    │
└──────────────┘

┌──────────────┐
│    Order     │──────► PostgreSQL (Orders DB)
│   Service    │
└──────────────┘
```

Each service would have its own database (Database per Service pattern).

## Communication Patterns

### Synchronous Communication

**Protocol**: HTTP/JSON

**Example - Cart Service → Product Service**:
```go
func (s *CartService) getProduct(productID string) (*models.Product, error) {
    url := fmt.Sprintf("%s/products/%s", s.productURL, productID)
    resp, err := http.Get(url)
    // Parse response
    var product models.Product
    json.NewDecoder(resp.Body).Decode(&product)
    return &product, nil
}
```

**Production Alternative**: gRPC for inter-service communication
- Better performance
- Type safety with Protocol Buffers
- Bi-directional streaming

### Asynchronous Communication (Future)

**Pattern**: Event-Driven Architecture with Message Queue

```
┌──────────────┐
│    Order     │
│   Service    │
└──────┬───────┘
       │
       │ Publish: OrderCreated
       ▼
┌──────────────────┐
│   Message Queue  │  (Kafka/RabbitMQ)
│   (Events)       │
└──────┬───────────┘
       │
       │ Subscribe: OrderCreated
       ▼
┌──────────────────┐
│  Notification    │
│   Service        │
└──────────────────┘
```

## Deployment Architecture

### Docker Deployment

**Pattern**: Containerization with orchestration

```yaml
services:
  gateway:     # API Gateway
  product:     # Product Service
  user:        # User Service
  cart:        # Cart Service
  order:       # Order Service
```

**Network**: Single Docker network (ecommerce-network)

**Port Mapping**:
- Gateway: 8080:8080 (exposed to host)
- Services: 8081-8084 (internal only in production)

### Kubernetes Deployment (Production)

```yaml
apiVersion: apps/v1
kind: Deployment
metadata:
  name: product-service
spec:
  replicas: 3
  selector:
    matchLabels:
      app: product
  template:
    metadata:
      labels:
        app: product
    spec:
      containers:
      - name: product
        image: ecommerce/product:latest
        ports:
        - containerPort: 8081
        env:
        - name: DATABASE_URL
          valueFrom:
            secretKeyRef:
              name: db-secrets
              key: url
        resources:
          requests:
            memory: "128Mi"
            cpu: "100m"
          limits:
            memory: "256Mi"
            cpu: "200m"
---
apiVersion: v1
kind: Service
metadata:
  name: product-service
spec:
  selector:
    app: product
  ports:
  - port: 8081
    targetPort: 8081
  type: ClusterIP
```

## Scaling Strategy

### Horizontal Scaling

Each service can be scaled independently:

```bash
# Scale product service to 5 instances
kubectl scale deployment product-service --replicas=5

# Scale order service to 3 instances
kubectl scale deployment order-service --replicas=3
```

### Load Balancing

**Current**: Docker Compose (round-robin)
**Production**: Kubernetes Service (load balancing built-in)

### Caching Strategy (Future)

```
┌──────────────┐
│   Product    │──────┐
│   Service    │      │
└──────────────┘      ├──► Redis (Product Cache)
                      │    - Hot products
                      │    - Search results
┌──────────────┐      │
│    Cart      │──────┘
│   Service    │
└──────────────┘
```

## Resilience Patterns

### Health Checks

Every service exposes `/health`:

```json
{
  "status": "healthy",
  "service": "product",
  "timestamp": "2025-10-21T13:00:00Z"
}
```

### Circuit Breaker (Future)

Protect services from cascade failures:

```go
type CircuitBreaker struct {
    maxFailures int
    timeout     time.Duration
    state       State  // Closed, Open, Half-Open
}
```

**States**:
- **Closed**: Normal operation
- **Open**: Fast fail (after threshold)
- **Half-Open**: Test recovery

### Retry Logic (Future)

```go
func retryWithBackoff(fn func() error, maxRetries int) error {
    for i := 0; i < maxRetries; i++ {
        err := fn()
        if err == nil {
            return nil
        }
        time.Sleep(time.Second * time.Duration(math.Pow(2, float64(i))))
    }
    return errors.New("max retries exceeded")
}
```

## Security Architecture

### Authentication Flow

```
1. Client → POST /api/auth/login (email, password)
2. User Service validates credentials
3. User Service generates JWT token
4. Client receives token
5. Client → GET /api/cart (Authorization: Bearer <token>)
6. Gateway validates token
7. Gateway adds X-User-ID header
8. Backend service uses user ID
```

### Authorization (Future)

**Pattern**: Role-Based Access Control (RBAC)

```go
type User struct {
    ID    string
    Email string
    Roles []string  // ["user", "admin"]
}

func (m *AuthMiddleware) RequireRole(role string) http.HandlerFunc {
    return func(w http.ResponseWriter, r *http.Request) {
        // Check if user has required role
    }
}
```

### Security Headers

```go
w.Header().Set("X-Content-Type-Options", "nosniff")
w.Header().Set("X-Frame-Options", "DENY")
w.Header().Set("X-XSS-Protection", "1; mode=block")
```

## Monitoring and Observability (Future)

### Metrics

**Tool**: Prometheus

```go
var (
    httpRequestsTotal = prometheus.NewCounterVec(
        prometheus.CounterOpts{
            Name: "http_requests_total",
            Help: "Total HTTP requests",
        },
        []string{"service", "method", "endpoint", "status"},
    )

    httpRequestDuration = prometheus.NewHistogramVec(
        prometheus.HistogramOpts{
            Name: "http_request_duration_seconds",
            Help: "HTTP request duration",
        },
        []string{"service", "method", "endpoint"},
    )
)
```

### Distributed Tracing

**Tool**: Jaeger/Zipkin

```go
func (s *Service) handleRequest(w http.ResponseWriter, r *http.Request) {
    span := opentracing.StartSpan("handle_request")
    defer span.Finish()

    // Add trace ID to context
    ctx := opentracing.ContextWithSpan(r.Context(), span)

    // Pass context to downstream services
}
```

### Logging Aggregation

**Tool**: ELK Stack (Elasticsearch, Logstash, Kibana)

```json
{
  "timestamp": "2025-10-21T13:00:00Z",
  "service": "product",
  "level": "INFO",
  "message": "Product created",
  "trace_id": "abc123",
  "user_id": "user_456"
}
```

## Data Flow Examples

### Example 1: User Registration and First Order

```
1. POST /api/auth/register
   Gateway → User Service
   User Service creates user
   Returns JWT token

2. GET /api/products
   Gateway → Product Service
   Product Service returns product list

3. POST /api/cart/items
   Gateway → Cart Service
   Cart Service → Product Service (validate product)
   Cart Service saves cart
   Returns updated cart

4. POST /api/orders
   Gateway → Order Service
   Order Service validates items
   Order Service creates order
   Returns order confirmation
```

### Example 2: Inter-Service Communication

```
Client
  │
  │ POST /api/cart/items
  ▼
API Gateway
  │
  │ Validate JWT
  │ Add X-User-ID header
  ▼
Cart Service
  │
  │ GET /products/{id}
  ▼
Product Service
  │
  │ Return product details
  ▼
Cart Service
  │
  │ Validate inventory
  │ Update cart
  ▼
API Gateway
  │
  ▼
Client
```

## Performance Considerations

### Current Performance

- Request latency: <10ms (in-memory)
- Throughput: Limited by single instance
- Concurrency: Go's goroutines handle concurrent requests

### Production Optimizations

1. **Database Connection Pooling**
```go
db, _ := sql.Open("postgres", connStr)
db.SetMaxOpenConns(25)
db.SetMaxIdleConns(5)
db.SetConnMaxLifetime(5 * time.Minute)
```

2. **Caching**
```go
// Redis cache
func (s *ProductService) GetProduct(id string) (*Product, error) {
    // Check cache
    if cached, err := s.cache.Get(id); err == nil {
        return cached, nil
    }

    // Load from database
    product, err := s.db.GetProduct(id)

    // Cache result
    s.cache.Set(id, product, 5*time.Minute)

    return product, err
}
```

3. **Database Indexes**
```sql
CREATE INDEX idx_products_category ON products(category);
CREATE INDEX idx_orders_user_id ON orders(user_id);
CREATE INDEX idx_orders_status ON orders(status);
```

## Evolution Path

### Phase 1: Current State ✓
- Microservices architecture
- HTTP/JSON communication
- In-memory storage
- Docker deployment

### Phase 2: Production-Ready Database
- PostgreSQL for each service
- Database migrations
- Connection pooling
- Transactions

### Phase 3: Enhanced Resilience
- Circuit breakers
- Retry logic with backoff
- Rate limiting
- Timeouts

### Phase 4: Observability
- Distributed tracing
- Metrics collection
- Centralized logging
- Health dashboards

### Phase 5: Event-Driven
- Message queue (Kafka/RabbitMQ)
- Async event processing
- Saga pattern for transactions
- Event sourcing

### Phase 6: Advanced Features
- gRPC communication
- Service mesh (Istio)
- API versioning
- GraphQL gateway

## References

**Patterns**:
- Microservices Architecture (Chris Richardson)
- API Gateway Pattern
- Database per Service
- Event-Driven Architecture
- Circuit Breaker Pattern
- Saga Pattern

**Tools**:
- Go standard library
- Docker & Docker Compose
- Kubernetes (recommended)
- PostgreSQL (recommended)
- Redis (recommended)
- Kafka (optional)

---

**Document Version**: 1.0
**Last Updated**: 2025-10-21
**Status**: Production-Ready Educational Implementation
