package main

import (
	"encoding/json"
	"fmt"
	"net/http"

	"github.com/moderngo/ecommerce-platform/internal/config"
	"github.com/moderngo/ecommerce-platform/internal/database"
	"github.com/moderngo/ecommerce-platform/internal/middleware"
	"github.com/moderngo/ecommerce-platform/internal/models"
	"github.com/moderngo/ecommerce-platform/pkg/logger"
)

type CartService struct {
	db         *database.InMemoryDB
	log        *logger.Logger
	productURL string
}

func main() {
	log := logger.New("[CART]")
	cfg := config.Load("cart")
	cfg.Port = config.GetPort("cart")

	db := database.NewInMemoryDB()
	service := &CartService{
		db:         db,
		log:        log,
		productURL: config.GetServiceURL("product"),
	}

	log.Info("Starting Cart Service on port %s", cfg.Port)

	mux := http.NewServeMux()

	// Health check
	mux.HandleFunc("/health", middleware.HealthCheck("cart"))

	// Cart endpoints
	mux.HandleFunc("/cart", service.handleCart)
	mux.HandleFunc("/cart/items", service.handleCartItems)

	// Apply middleware
	handler := middleware.Recovery(log)(
		middleware.Logger(log)(mux),
	)

	log.Info("Cart Service listening on http://localhost:%s", cfg.Port)
	if err := http.ListenAndServe(":"+cfg.Port, handler); err != nil {
		log.Fatal("Failed to start server: %v", err)
	}
}

func (s *CartService) handleCart(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodGet {
		respondError(w, http.StatusMethodNotAllowed, "Method not allowed")
		return
	}

	// Get user ID from header (set by gateway auth middleware)
	userID := r.Header.Get("X-User-ID")
	if userID == "" {
		userID = "guest" // Allow guest carts
	}

	cart, err := s.db.GetCart(userID)
	if err != nil {
		s.log.Error("Failed to get cart: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to retrieve cart")
		return
	}

	respondJSON(w, http.StatusOK, cart)
}

func (s *CartService) handleCartItems(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		respondError(w, http.StatusMethodNotAllowed, "Method not allowed")
		return
	}

	// Get user ID
	userID := r.Header.Get("X-User-ID")
	if userID == "" {
		userID = "guest"
	}

	// Parse request
	var req struct {
		ProductID string `json:"product_id"`
		Quantity  int    `json:"quantity"`
	}
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		respondError(w, http.StatusBadRequest, "Invalid request body")
		return
	}

	if req.ProductID == "" || req.Quantity <= 0 {
		respondError(w, http.StatusBadRequest, "Valid product_id and quantity required")
		return
	}

	// Get product details from product service
	product, err := s.getProduct(req.ProductID)
	if err != nil {
		s.log.Error("Failed to get product: %v", err)
		respondError(w, http.StatusNotFound, "Product not found")
		return
	}

	// Check inventory
	if product.Inventory < req.Quantity {
		respondError(w, http.StatusBadRequest, "Insufficient inventory")
		return
	}

	// Get current cart
	cart, err := s.db.GetCart(userID)
	if err != nil {
		s.log.Error("Failed to get cart: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to retrieve cart")
		return
	}

	// Add or update item in cart
	found := false
	for i, item := range cart.Items {
		if item.ProductID == req.ProductID {
			cart.Items[i].Quantity += req.Quantity
			found = true
			break
		}
	}

	if !found {
		cart.Items = append(cart.Items, models.CartItem{
			ProductID: product.ID,
			Name:      product.Name,
			Price:     product.Price,
			Quantity:  req.Quantity,
		})
	}

	// Recalculate total
	cart.Total = 0
	for _, item := range cart.Items {
		cart.Total += item.Price * float64(item.Quantity)
	}

	// Save cart
	if err := s.db.SaveCart(cart); err != nil {
		s.log.Error("Failed to save cart: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to save cart")
		return
	}

	s.log.Info("Added item to cart for user %s: %s x%d", userID, product.ID, req.Quantity)
	respondJSON(w, http.StatusOK, cart)
}

// getProduct fetches product details from product service
func (s *CartService) getProduct(productID string) (*models.Product, error) {
	url := fmt.Sprintf("%s/products/%s", s.productURL, productID)
	resp, err := http.Get(url)
	if err != nil {
		return nil, err
	}
	defer resp.Body.Close()

	if resp.StatusCode != http.StatusOK {
		return nil, fmt.Errorf("product service returned status %d", resp.StatusCode)
	}

	var product models.Product
	if err := json.NewDecoder(resp.Body).Decode(&product); err != nil {
		return nil, err
	}

	return &product, nil
}

func respondJSON(w http.ResponseWriter, status int, data interface{}) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)
	json.NewEncoder(w).Encode(data)
}

func respondError(w http.ResponseWriter, status int, message string) {
	respondJSON(w, status, models.ErrorResponse{Error: message})
}
