package main

import (
	"bytes"
	"encoding/json"
	"io"
	"net/http"

	"github.com/moderngo/ecommerce-platform/internal/config"
	"github.com/moderngo/ecommerce-platform/internal/middleware"
	"github.com/moderngo/ecommerce-platform/pkg/logger"
)

func main() {
	log := logger.New("[GATEWAY]")
	cfg := config.Load("gateway")
	cfg.Port = config.GetPort("gateway")

	log.Info("Starting API Gateway on port %s", cfg.Port)

	mux := http.NewServeMux()

	// Health check
	mux.HandleFunc("/health", middleware.HealthCheck("gateway"))

	// API routes - proxy to microservices
	mux.HandleFunc("/api/products", handleProducts(cfg, log))
	mux.HandleFunc("/api/products/", handleProductByID(cfg, log))

	mux.HandleFunc("/api/auth/register", handleAuthRegister(cfg, log))
	mux.HandleFunc("/api/auth/login", handleAuthLogin(cfg, log))

	mux.HandleFunc("/api/cart", handleCart(cfg, log))
	mux.HandleFunc("/api/cart/items", handleCartItems(cfg, log))

	mux.HandleFunc("/api/orders", handleOrders(cfg, log))
	mux.HandleFunc("/api/orders/", handleOrderByID(cfg, log))

	// Apply middleware
	handler := middleware.CORS(
		middleware.Recovery(log)(
			middleware.Logger(log)(mux),
		),
	)

	log.Info("API Gateway listening on http://localhost:%s", cfg.Port)
	log.Info("Endpoints:")
	log.Info("  GET  /health")
	log.Info("  GET  /api/products")
	log.Info("  POST /api/auth/register")
	log.Info("  POST /api/auth/login")
	log.Info("  GET  /api/cart")
	log.Info("  POST /api/cart/items")
	log.Info("  POST /api/orders")
	log.Info("  GET  /api/orders")

	if err := http.ListenAndServe(":"+cfg.Port, handler); err != nil {
		log.Fatal("Failed to start server: %v", err)
	}
}

// Product handlers - proxy to product service

func handleProducts(cfg *config.Config, log *logger.Logger) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		serviceURL := config.GetServiceURL("product")
		proxyRequest(w, r, serviceURL+"/products", log)
	}
}

func handleProductByID(cfg *config.Config, log *logger.Logger) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		serviceURL := config.GetServiceURL("product")
		proxyRequest(w, r, serviceURL+r.URL.Path[len("/api"):], log)
	}
}

// Auth handlers - proxy to user service

func handleAuthRegister(cfg *config.Config, log *logger.Logger) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		serviceURL := config.GetServiceURL("user")
		proxyRequest(w, r, serviceURL+"/auth/register", log)
	}
}

func handleAuthLogin(cfg *config.Config, log *logger.Logger) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		serviceURL := config.GetServiceURL("user")
		proxyRequest(w, r, serviceURL+"/auth/login", log)
	}
}

// Cart handlers - proxy to cart service

func handleCart(cfg *config.Config, log *logger.Logger) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		serviceURL := config.GetServiceURL("cart")
		proxyRequest(w, r, serviceURL+"/cart", log)
	}
}

func handleCartItems(cfg *config.Config, log *logger.Logger) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		serviceURL := config.GetServiceURL("cart")
		proxyRequest(w, r, serviceURL+"/cart/items", log)
	}
}

// Order handlers - proxy to order service

func handleOrders(cfg *config.Config, log *logger.Logger) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		serviceURL := config.GetServiceURL("order")
		proxyRequest(w, r, serviceURL+"/orders", log)
	}
}

func handleOrderByID(cfg *config.Config, log *logger.Logger) http.HandlerFunc {
	return func(w http.ResponseWriter, r *http.Request) {
		serviceURL := config.GetServiceURL("order")
		proxyRequest(w, r, serviceURL+r.URL.Path[len("/api"):], log)
	}
}

// proxyRequest forwards requests to backend services
func proxyRequest(w http.ResponseWriter, r *http.Request, targetURL string, log *logger.Logger) {
	// Read request body
	var bodyBytes []byte
	if r.Body != nil {
		bodyBytes, _ = io.ReadAll(r.Body)
		r.Body.Close()
	}

	// Create new request
	req, err := http.NewRequest(r.Method, targetURL, bytes.NewBuffer(bodyBytes))
	if err != nil {
		log.Error("Failed to create proxy request: %v", err)
		respondJSON(w, http.StatusInternalServerError, map[string]string{"error": "Internal server error"})
		return
	}

	// Copy headers
	for key, values := range r.Header {
		for _, value := range values {
			req.Header.Add(key, value)
		}
	}

	// Forward request
	client := &http.Client{}
	resp, err := client.Do(req)
	if err != nil {
		log.Error("Failed to proxy request to %s: %v", targetURL, err)
		respondJSON(w, http.StatusServiceUnavailable, map[string]string{"error": "Service unavailable"})
		return
	}
	defer resp.Body.Close()

	// Copy response headers
	for key, values := range resp.Header {
		for _, value := range values {
			w.Header().Add(key, value)
		}
	}

	// Copy response body
	w.WriteHeader(resp.StatusCode)
	io.Copy(w, resp.Body)
}

func respondJSON(w http.ResponseWriter, status int, data interface{}) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)
	json.NewEncoder(w).Encode(data)
}
