package main

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/moderngo/ecommerce-platform/internal/config"
	"github.com/moderngo/ecommerce-platform/internal/database"
	"github.com/moderngo/ecommerce-platform/internal/middleware"
	"github.com/moderngo/ecommerce-platform/internal/models"
	"github.com/moderngo/ecommerce-platform/pkg/logger"
)

type OrderService struct {
	db      *database.InMemoryDB
	log     *logger.Logger
	cartURL string
}

func main() {
	log := logger.New("[ORDER]")
	cfg := config.Load("order")
	cfg.Port = config.GetPort("order")

	db := database.NewInMemoryDB()
	service := &OrderService{
		db:      db,
		log:     log,
		cartURL: config.GetServiceURL("cart"),
	}

	log.Info("Starting Order Service on port %s", cfg.Port)

	mux := http.NewServeMux()

	// Health check
	mux.HandleFunc("/health", middleware.HealthCheck("order"))

	// Order endpoints
	mux.HandleFunc("/orders", service.handleOrders)
	mux.HandleFunc("/orders/", service.handleOrderByID)

	// Apply middleware
	handler := middleware.Recovery(log)(
		middleware.Logger(log)(mux),
	)

	log.Info("Order Service listening on http://localhost:%s", cfg.Port)
	if err := http.ListenAndServe(":"+cfg.Port, handler); err != nil {
		log.Fatal("Failed to start server: %v", err)
	}
}

func (s *OrderService) handleOrders(w http.ResponseWriter, r *http.Request) {
	switch r.Method {
	case http.MethodGet:
		s.listOrders(w, r)
	case http.MethodPost:
		s.createOrder(w, r)
	default:
		respondError(w, http.StatusMethodNotAllowed, "Method not allowed")
	}
}

func (s *OrderService) handleOrderByID(w http.ResponseWriter, r *http.Request) {
	// Extract order ID from path /orders/{id}
	id := strings.TrimPrefix(r.URL.Path, "/orders/")
	if id == "" {
		respondError(w, http.StatusBadRequest, "Order ID required")
		return
	}

	if r.Method != http.MethodGet {
		respondError(w, http.StatusMethodNotAllowed, "Method not allowed")
		return
	}

	s.getOrder(w, r, id)
}

func (s *OrderService) listOrders(w http.ResponseWriter, r *http.Request) {
	// Get user ID
	userID := r.Header.Get("X-User-ID")
	if userID == "" {
		respondError(w, http.StatusUnauthorized, "User not authenticated")
		return
	}

	orders, err := s.db.ListOrdersByUser(userID)
	if err != nil {
		s.log.Error("Failed to list orders: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to retrieve orders")
		return
	}

	respondJSON(w, http.StatusOK, orders)
}

func (s *OrderService) getOrder(w http.ResponseWriter, r *http.Request, id string) {
	order, err := s.db.GetOrder(id)
	if err != nil {
		if err == database.ErrNotFound {
			respondError(w, http.StatusNotFound, "Order not found")
			return
		}
		s.log.Error("Failed to get order: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to retrieve order")
		return
	}

	respondJSON(w, http.StatusOK, order)
}

func (s *OrderService) createOrder(w http.ResponseWriter, r *http.Request) {
	// Get user ID
	userID := r.Header.Get("X-User-ID")
	if userID == "" {
		userID = "guest"
	}

	// Parse request
	var req struct {
		Items           []models.OrderItem `json:"items"`
		ShippingAddress models.Address     `json:"shipping_address"`
		PaymentMethod   string             `json:"payment_method"`
	}
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		respondError(w, http.StatusBadRequest, "Invalid request body")
		return
	}

	// Validate request
	if len(req.Items) == 0 {
		respondError(w, http.StatusBadRequest, "Order must contain at least one item")
		return
	}

	if req.ShippingAddress.Street == "" || req.ShippingAddress.City == "" {
		respondError(w, http.StatusBadRequest, "Valid shipping address required")
		return
	}

	if req.PaymentMethod == "" {
		respondError(w, http.StatusBadRequest, "Payment method required")
		return
	}

	// Calculate total
	var total float64
	for _, item := range req.Items {
		total += item.Price * float64(item.Quantity)
	}

	// Create order
	order := &models.Order{
		ID:              fmt.Sprintf("order_%d", time.Now().UnixNano()),
		UserID:          userID,
		Items:           req.Items,
		Total:           total,
		Status:          "pending",
		ShippingAddress: req.ShippingAddress,
		PaymentMethod:   req.PaymentMethod,
		CreatedAt:       time.Now(),
		UpdatedAt:       time.Now(),
	}

	if err := s.db.CreateOrder(order); err != nil {
		s.log.Error("Failed to create order: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to create order")
		return
	}

	s.log.Info("Created order: %s for user %s, total: $%.2f", order.ID, userID, total)

	// In a real system, this would:
	// 1. Validate inventory
	// 2. Process payment
	// 3. Reserve inventory
	// 4. Clear user's cart
	// 5. Send confirmation notification
	// 6. Publish order created event

	// For demo, just update status
	order.Status = "confirmed"
	order.UpdatedAt = time.Now()
	s.db.UpdateOrder(order)

	respondJSON(w, http.StatusCreated, order)
}

func respondJSON(w http.ResponseWriter, status int, data interface{}) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)
	json.NewEncoder(w).Encode(data)
}

func respondError(w http.ResponseWriter, status int, message string) {
	respondJSON(w, status, models.ErrorResponse{Error: message})
}
