package main

import (
	"encoding/json"
	"fmt"
	"net/http"
	"strings"
	"time"

	"github.com/moderngo/ecommerce-platform/internal/config"
	"github.com/moderngo/ecommerce-platform/internal/database"
	"github.com/moderngo/ecommerce-platform/internal/middleware"
	"github.com/moderngo/ecommerce-platform/internal/models"
	"github.com/moderngo/ecommerce-platform/pkg/logger"
)

type ProductService struct {
	db  *database.InMemoryDB
	log *logger.Logger
}

func main() {
	log := logger.New("[PRODUCT]")
	cfg := config.Load("product")
	cfg.Port = config.GetPort("product")

	db := database.NewInMemoryDB()
	service := &ProductService{db: db, log: log}

	// Seed some sample products
	service.seedProducts()

	log.Info("Starting Product Service on port %s", cfg.Port)

	mux := http.NewServeMux()

	// Health check
	mux.HandleFunc("/health", middleware.HealthCheck("product"))

	// Product endpoints
	mux.HandleFunc("/products", service.handleProducts)
	mux.HandleFunc("/products/", service.handleProductByID)

	// Apply middleware
	handler := middleware.Recovery(log)(
		middleware.Logger(log)(mux),
	)

	log.Info("Product Service listening on http://localhost:%s", cfg.Port)
	if err := http.ListenAndServe(":"+cfg.Port, handler); err != nil {
		log.Fatal("Failed to start server: %v", err)
	}
}

func (s *ProductService) handleProducts(w http.ResponseWriter, r *http.Request) {
	switch r.Method {
	case http.MethodGet:
		s.listProducts(w, r)
	case http.MethodPost:
		s.createProduct(w, r)
	default:
		respondError(w, http.StatusMethodNotAllowed, "Method not allowed")
	}
}

func (s *ProductService) handleProductByID(w http.ResponseWriter, r *http.Request) {
	// Extract product ID from path /products/{id}
	id := strings.TrimPrefix(r.URL.Path, "/products/")
	if id == "" {
		respondError(w, http.StatusBadRequest, "Product ID required")
		return
	}

	switch r.Method {
	case http.MethodGet:
		s.getProduct(w, r, id)
	case http.MethodPut:
		s.updateProduct(w, r, id)
	case http.MethodDelete:
		s.deleteProduct(w, r, id)
	default:
		respondError(w, http.StatusMethodNotAllowed, "Method not allowed")
	}
}

func (s *ProductService) listProducts(w http.ResponseWriter, r *http.Request) {
	products, err := s.db.ListProducts()
	if err != nil {
		s.log.Error("Failed to list products: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to retrieve products")
		return
	}

	respondJSON(w, http.StatusOK, products)
}

func (s *ProductService) getProduct(w http.ResponseWriter, r *http.Request, id string) {
	product, err := s.db.GetProduct(id)
	if err != nil {
		if err == database.ErrNotFound {
			respondError(w, http.StatusNotFound, "Product not found")
			return
		}
		s.log.Error("Failed to get product: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to retrieve product")
		return
	}

	respondJSON(w, http.StatusOK, product)
}

func (s *ProductService) createProduct(w http.ResponseWriter, r *http.Request) {
	var product models.Product
	if err := json.NewDecoder(r.Body).Decode(&product); err != nil {
		respondError(w, http.StatusBadRequest, "Invalid request body")
		return
	}

	// Generate ID
	product.ID = fmt.Sprintf("prod_%d", time.Now().UnixNano())

	if err := s.db.CreateProduct(&product); err != nil {
		s.log.Error("Failed to create product: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to create product")
		return
	}

	s.log.Info("Created product: %s", product.ID)
	respondJSON(w, http.StatusCreated, product)
}

func (s *ProductService) updateProduct(w http.ResponseWriter, r *http.Request, id string) {
	var product models.Product
	if err := json.NewDecoder(r.Body).Decode(&product); err != nil {
		respondError(w, http.StatusBadRequest, "Invalid request body")
		return
	}

	product.ID = id
	if err := s.db.UpdateProduct(&product); err != nil {
		if err == database.ErrNotFound {
			respondError(w, http.StatusNotFound, "Product not found")
			return
		}
		s.log.Error("Failed to update product: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to update product")
		return
	}

	s.log.Info("Updated product: %s", product.ID)
	respondJSON(w, http.StatusOK, product)
}

func (s *ProductService) deleteProduct(w http.ResponseWriter, r *http.Request, id string) {
	if err := s.db.DeleteProduct(id); err != nil {
		if err == database.ErrNotFound {
			respondError(w, http.StatusNotFound, "Product not found")
			return
		}
		s.log.Error("Failed to delete product: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to delete product")
		return
	}

	s.log.Info("Deleted product: %s", id)
	respondJSON(w, http.StatusOK, map[string]bool{"success": true})
}

func (s *ProductService) seedProducts() {
	products := []*models.Product{
		{
			ID:          "prod_1",
			Name:        "Wireless Mouse",
			Description: "Ergonomic wireless mouse with USB receiver",
			Price:       29.99,
			SKU:         "MOUSE-001",
			Inventory:   100,
			Category:    "Electronics",
		},
		{
			ID:          "prod_2",
			Name:        "Mechanical Keyboard",
			Description: "RGB backlit mechanical keyboard with blue switches",
			Price:       89.99,
			SKU:         "KB-001",
			Inventory:   50,
			Category:    "Electronics",
		},
		{
			ID:          "prod_3",
			Name:        "USB-C Cable",
			Description: "Premium braided USB-C to USB-C cable, 2 meters",
			Price:       14.99,
			SKU:         "CABLE-001",
			Inventory:   200,
			Category:    "Accessories",
		},
		{
			ID:          "prod_4",
			Name:        "Laptop Stand",
			Description: "Adjustable aluminum laptop stand with cooling",
			Price:       49.99,
			SKU:         "STAND-001",
			Inventory:   75,
			Category:    "Accessories",
		},
		{
			ID:          "prod_5",
			Name:        "Webcam HD",
			Description: "1080p HD webcam with built-in microphone",
			Price:       69.99,
			SKU:         "CAM-001",
			Inventory:   60,
			Category:    "Electronics",
		},
	}

	for _, product := range products {
		s.db.CreateProduct(product)
		s.log.Info("Seeded product: %s - %s", product.ID, product.Name)
	}
}

func respondJSON(w http.ResponseWriter, status int, data interface{}) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)
	json.NewEncoder(w).Encode(data)
}

func respondError(w http.ResponseWriter, status int, message string) {
	respondJSON(w, status, models.ErrorResponse{Error: message})
}
