package main

import (
	"crypto/sha256"
	"encoding/hex"
	"encoding/json"
	"fmt"
	"net/http"
	"time"

	"github.com/moderngo/ecommerce-platform/internal/config"
	"github.com/moderngo/ecommerce-platform/internal/database"
	"github.com/moderngo/ecommerce-platform/internal/middleware"
	"github.com/moderngo/ecommerce-platform/internal/models"
	"github.com/moderngo/ecommerce-platform/pkg/logger"
)

type UserService struct {
	db        *database.InMemoryDB
	log       *logger.Logger
	jwtSecret string
}

func main() {
	log := logger.New("[USER]")
	cfg := config.Load("user")
	cfg.Port = config.GetPort("user")

	db := database.NewInMemoryDB()
	service := &UserService{
		db:        db,
		log:       log,
		jwtSecret: cfg.JWTSecret,
	}

	log.Info("Starting User Service on port %s", cfg.Port)

	mux := http.NewServeMux()

	// Health check
	mux.HandleFunc("/health", middleware.HealthCheck("user"))

	// Auth endpoints
	mux.HandleFunc("/auth/register", service.handleRegister)
	mux.HandleFunc("/auth/login", service.handleLogin)

	// Apply middleware
	handler := middleware.Recovery(log)(
		middleware.Logger(log)(mux),
	)

	log.Info("User Service listening on http://localhost:%s", cfg.Port)
	if err := http.ListenAndServe(":"+cfg.Port, handler); err != nil {
		log.Fatal("Failed to start server: %v", err)
	}
}

func (s *UserService) handleRegister(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		respondError(w, http.StatusMethodNotAllowed, "Method not allowed")
		return
	}

	var req models.AuthRequest
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		respondError(w, http.StatusBadRequest, "Invalid request body")
		return
	}

	// Validate input
	if req.Email == "" || req.Password == "" || req.Name == "" {
		respondError(w, http.StatusBadRequest, "Email, password, and name are required")
		return
	}

	// Check if user already exists
	if _, err := s.db.GetUserByEmail(req.Email); err == nil {
		respondError(w, http.StatusConflict, "User already exists")
		return
	}

	// Hash password (simplified - use bcrypt in production)
	hashedPassword := hashPassword(req.Password)

	// Create user
	user := &models.User{
		ID:        fmt.Sprintf("user_%d", time.Now().UnixNano()),
		Email:     req.Email,
		Password:  hashedPassword,
		Name:      req.Name,
		CreatedAt: time.Now(),
	}

	if err := s.db.CreateUser(user); err != nil {
		s.log.Error("Failed to create user: %v", err)
		respondError(w, http.StatusInternalServerError, "Failed to create user")
		return
	}

	s.log.Info("User registered: %s (%s)", user.ID, user.Email)

	// Generate token
	token := s.generateToken(user.ID)
	expiresAt := time.Now().Add(24 * time.Hour)

	respondJSON(w, http.StatusCreated, models.AuthResponse{
		Token:     token,
		UserID:    user.ID,
		ExpiresAt: expiresAt,
	})
}

func (s *UserService) handleLogin(w http.ResponseWriter, r *http.Request) {
	if r.Method != http.MethodPost {
		respondError(w, http.StatusMethodNotAllowed, "Method not allowed")
		return
	}

	var req models.AuthRequest
	if err := json.NewDecoder(r.Body).Decode(&req); err != nil {
		respondError(w, http.StatusBadRequest, "Invalid request body")
		return
	}

	// Validate input
	if req.Email == "" || req.Password == "" {
		respondError(w, http.StatusBadRequest, "Email and password are required")
		return
	}

	// Get user by email
	user, err := s.db.GetUserByEmail(req.Email)
	if err != nil {
		respondError(w, http.StatusUnauthorized, "Invalid credentials")
		return
	}

	// Verify password
	if !verifyPassword(user.Password, req.Password) {
		respondError(w, http.StatusUnauthorized, "Invalid credentials")
		return
	}

	s.log.Info("User logged in: %s (%s)", user.ID, user.Email)

	// Generate token
	token := s.generateToken(user.ID)
	expiresAt := time.Now().Add(24 * time.Hour)

	respondJSON(w, http.StatusOK, models.AuthResponse{
		Token:     token,
		UserID:    user.ID,
		ExpiresAt: expiresAt,
	})
}

// generateToken generates a simple token (use JWT in production)
func (s *UserService) generateToken(userID string) string {
	// Simplified token generation - in production use proper JWT library
	data := fmt.Sprintf("%s:%s:%d", userID, s.jwtSecret, time.Now().Unix())
	hash := sha256.Sum256([]byte(data))
	return hex.EncodeToString(hash[:])
}

// hashPassword hashes a password (simplified - use bcrypt in production)
func hashPassword(password string) string {
	hash := sha256.Sum256([]byte(password))
	return hex.EncodeToString(hash[:])
}

// verifyPassword verifies a password against a hash
func verifyPassword(hashedPassword, password string) bool {
	return hashedPassword == hashPassword(password)
}

func respondJSON(w http.ResponseWriter, status int, data interface{}) {
	w.Header().Set("Content-Type", "application/json")
	w.WriteHeader(status)
	json.NewEncoder(w).Encode(data)
}

func respondError(w http.ResponseWriter, status int, message string) {
	respondJSON(w, status, models.ErrorResponse{Error: message})
}
