package config

import (
	"fmt"
	"os"
	"strconv"
)

// Config holds application configuration
type Config struct {
	ServiceName string
	Port        string
	JWTSecret   string
	DatabaseURL string

	// Service discovery (simplified - just URLs)
	GatewayURL  string
	ProductURL  string
	UserURL     string
	CartURL     string
	OrderURL    string
}

// Load loads configuration from environment variables with defaults
func Load(serviceName string) *Config {
	return &Config{
		ServiceName: serviceName,
		Port:        getEnv("PORT", "8080"),
		JWTSecret:   getEnv("JWT_SECRET", "supersecretkey-change-in-production"),
		DatabaseURL: getEnv("DATABASE_URL", "postgres://user:pass@localhost:5432/ecommerce?sslmode=disable"),

		// Service URLs for inter-service communication
		GatewayURL:  getEnv("GATEWAY_URL", "http://localhost:8080"),
		ProductURL:  getEnv("PRODUCT_URL", "http://localhost:8081"),
		UserURL:     getEnv("USER_URL", "http://localhost:8082"),
		CartURL:     getEnv("CART_URL", "http://localhost:8083"),
		OrderURL:    getEnv("ORDER_URL", "http://localhost:8084"),
	}
}

// GetPort returns the port for a specific service
func GetPort(service string) string {
	ports := map[string]string{
		"gateway": "8080",
		"product": "8081",
		"user":    "8082",
		"cart":    "8083",
		"order":   "8084",
	}

	if envPort := os.Getenv(fmt.Sprintf("%s_PORT", service)); envPort != "" {
		return envPort
	}

	return ports[service]
}

// GetServiceURL returns the base URL for a service
func GetServiceURL(service string) string {
	port := GetPort(service)
	return fmt.Sprintf("http://localhost:%s", port)
}

func getEnv(key, defaultValue string) string {
	if value := os.Getenv(key); value != "" {
		return value
	}
	return defaultValue
}

func getEnvAsInt(key string, defaultValue int) int {
	if value := os.Getenv(key); value != "" {
		if intVal, err := strconv.Atoi(value); err == nil {
			return intVal
		}
	}
	return defaultValue
}
