package database

import (
	"sync"

	"github.com/moderngo/ecommerce-platform/internal/models"
)

// InMemoryDB is a simple in-memory database for demonstration
// In production, replace with PostgreSQL/MySQL
type InMemoryDB struct {
	mu       sync.RWMutex
	users    map[string]*models.User
	products map[string]*models.Product
	carts    map[string]*models.Cart
	orders   map[string]*models.Order
}

// NewInMemoryDB creates a new in-memory database
func NewInMemoryDB() *InMemoryDB {
	return &InMemoryDB{
		users:    make(map[string]*models.User),
		products: make(map[string]*models.Product),
		carts:    make(map[string]*models.Cart),
		orders:   make(map[string]*models.Order),
	}
}

// User operations

func (db *InMemoryDB) CreateUser(user *models.User) error {
	db.mu.Lock()
	defer db.mu.Unlock()
	db.users[user.ID] = user
	return nil
}

func (db *InMemoryDB) GetUser(id string) (*models.User, error) {
	db.mu.RLock()
	defer db.mu.RUnlock()
	if user, ok := db.users[id]; ok {
		return user, nil
	}
	return nil, ErrNotFound
}

func (db *InMemoryDB) GetUserByEmail(email string) (*models.User, error) {
	db.mu.RLock()
	defer db.mu.RUnlock()
	for _, user := range db.users {
		if user.Email == email {
			return user, nil
		}
	}
	return nil, ErrNotFound
}

// Product operations

func (db *InMemoryDB) CreateProduct(product *models.Product) error {
	db.mu.Lock()
	defer db.mu.Unlock()
	db.products[product.ID] = product
	return nil
}

func (db *InMemoryDB) GetProduct(id string) (*models.Product, error) {
	db.mu.RLock()
	defer db.mu.RUnlock()
	if product, ok := db.products[id]; ok {
		return product, nil
	}
	return nil, ErrNotFound
}

func (db *InMemoryDB) ListProducts() ([]*models.Product, error) {
	db.mu.RLock()
	defer db.mu.RUnlock()
	products := make([]*models.Product, 0, len(db.products))
	for _, product := range db.products {
		products = append(products, product)
	}
	return products, nil
}

func (db *InMemoryDB) UpdateProduct(product *models.Product) error {
	db.mu.Lock()
	defer db.mu.Unlock()
	if _, ok := db.products[product.ID]; !ok {
		return ErrNotFound
	}
	db.products[product.ID] = product
	return nil
}

func (db *InMemoryDB) DeleteProduct(id string) error {
	db.mu.Lock()
	defer db.mu.Unlock()
	if _, ok := db.products[id]; !ok {
		return ErrNotFound
	}
	delete(db.products, id)
	return nil
}

// Cart operations

func (db *InMemoryDB) GetCart(userID string) (*models.Cart, error) {
	db.mu.RLock()
	defer db.mu.RUnlock()
	if cart, ok := db.carts[userID]; ok {
		return cart, nil
	}
	// Return empty cart if not found
	return &models.Cart{UserID: userID, Items: []models.CartItem{}, Total: 0}, nil
}

func (db *InMemoryDB) SaveCart(cart *models.Cart) error {
	db.mu.Lock()
	defer db.mu.Unlock()
	db.carts[cart.UserID] = cart
	return nil
}

func (db *InMemoryDB) ClearCart(userID string) error {
	db.mu.Lock()
	defer db.mu.Unlock()
	delete(db.carts, userID)
	return nil
}

// Order operations

func (db *InMemoryDB) CreateOrder(order *models.Order) error {
	db.mu.Lock()
	defer db.mu.Unlock()
	db.orders[order.ID] = order
	return nil
}

func (db *InMemoryDB) GetOrder(id string) (*models.Order, error) {
	db.mu.RLock()
	defer db.mu.RUnlock()
	if order, ok := db.orders[id]; ok {
		return order, nil
	}
	return nil, ErrNotFound
}

func (db *InMemoryDB) ListOrdersByUser(userID string) ([]*models.Order, error) {
	db.mu.RLock()
	defer db.mu.RUnlock()
	orders := make([]*models.Order, 0)
	for _, order := range db.orders {
		if order.UserID == userID {
			orders = append(orders, order)
		}
	}
	return orders, nil
}

func (db *InMemoryDB) UpdateOrder(order *models.Order) error {
	db.mu.Lock()
	defer db.mu.Unlock()
	if _, ok := db.orders[order.ID]; !ok {
		return ErrNotFound
	}
	db.orders[order.ID] = order
	return nil
}

// Errors
var (
	ErrNotFound = &DBError{Message: "record not found"}
)

type DBError struct {
	Message string
}

func (e *DBError) Error() string {
	return e.Message
}
